function makeScript(obj, filename, funcName)

    [stateNames, auxNames, ctrlNames, ...
        paramNames, inputNames] = getFieldNames(obj);

    fid = fopen(filename, 'w');

    fprintf(fid, ['function ' funcName '(obj, func, options)\n\n']);
    
    fprintf(fid, ['if ~exist(''options'',''var'')\n' ...
        '\t options = [];\n' ...
        'end\n\n']);
    
    fprintf(fid, ['[stateNames, auxNames, ctrlNames, ...\n'...
        '\t paramNames, inputNames] = getFieldNames(obj);\n']);

    %% Set time variable
    fprintf(fid, ['\n%% Set time variable\n' ...
    'time = obj.t.val;\n']);

    %% Convert inputs to matrix
    fprintf(fid, '\n%% Convert inputs to matrix\n');
    fprintf(fid, 'd = obj.d.(inputNames{1}).val(:,1);\n');
    fprintf(fid, 'for k=1:length(inputNames)\n');
    fprintf(fid, '\t d = [d obj.d.(inputNames{k}).val(:,2)];\n');
    fprintf(fid, 'end\n');
    
    %% Convert predefined controls to matrix
    fprintf(fid, '\n%% Convert predefined controls to matrix\n');
    fprintf(fid, 'u = [];\n');
    fprintf(fid, 'uTime = [];\n');
    fprintf(fid, 'for k=1:length(ctrlNames)\n');
    fprintf(fid, ['\t if isnumeric(obj.u.(ctrlNames{k}).val) && ~isscalar(obj.u.(ctrlNames{k}).val)\n' ...
        '\t\t uTime = obj.u.(ctrlNames{k}).val(:,1);\n' ...
        '\t\t u = [u obj.u.(ctrlNames{k}).val(:,2)];\n' ...
        '\t else\n' ...
            '\t\t if isempty(u)\n'...
                '\t\t\t u = nan(1,1);\n' ...
            '\t\t else\n' ...
                '\t\t\t u = [u nan(length(u(:,1)),1)];\n' ...
            '\t\t end\n' ...
        '\t end\n']);
    fprintf(fid, 'end\n');
    fprintf(fid, ['if ~isempty(uTime)\n' ...
        '\t u = [uTime u];\n' ...
        'end\n']);

    % create array of parameters
    fprintf(fid, '\n%% Create array of parameters\n');
    for k=1:length(paramNames)
        fprintf(fid, ['p(' num2str(k) ') = ' ...
            num2str(obj.p.(paramNames{k}).val) ';\n']);
    end

    
    %% Run simulation
    fprintf(fid, '\n%% Run simulation\n');
    fprintf(fid, '[t,x] = feval(func, @(t,x) ode(x, t, d, u, p), time, getInitialStates(obj), options);\n');

    fprintf(fid, 'setSolution(obj, t, x);\n\n');
    
    fprintf(fid, 'end\n');
    %% Function for the ODE solver
    fprintf(fid, '\n%% Function for the ODE solver\n');
    fprintf(fid, 'function dx = ode(x, t, d, u, p)\n\n');
    
    % sample inputs at time t
    fprintf(fid, ['\t %% sample inputs at time t\n' ...
    '\t if t < d(1,1)\n' ...
        '\t\t dSample = d(1,2:end);\n' ...
    '\t elseif t > d(end,1)\n' ...
        '\t\t dSample = d(end,2:end);\n' ...
    '\t else \n' ...
    '\t\t dSample = nan(size(d,2)-1,1);\n' ...
    '\t\t for k=1:(size(d,2)-1)\n' ...
        '\t\t\t dSample(k) = interp1(d(:,1),d(:,k+1),t);\n'...
    '\t\t end\n' ...
    '\t end\n' ...
    '\t d = dSample;\n']);

    % sample predefined controls at time t
    fprintf(fid, ['\n\t %% sample predefined controls at time t\n' ...
    '\t if t < u(1,1)\n' ...
        '\t\t uSample = u(1,2:end);\n' ...
    '\t elseif t > u(end,1)\n' ...
        '\t\t uSample = u(end,2:end);\n' ...
    '\t else \n' ...
    '\t\t uSample = nan(size(u,2)-1,1);\n' ...
    '\t\t for k=1:(size(u,2)-1)\n' ...
        '\t\t\t if ~isnan(u(1,k+1))\n' ...
            '\t\t\t\t uSample(k) = interp1(u(:,1),u(:,k+1),t);\n'...
        '\t\t\t end\n' ...
    '\t\t end\n' ...
    '\t end\n' ...
    '\t u = uSample;\n']);
    
    % calculate values of rule based controls
    fprintf(fid, '\n\t%% Calculate values of rule based controls\n');
    for k=1:length(ctrlNames)
        if ~(isnumeric(obj.u.(ctrlNames{k}).val) && ~isscalar(obj.u.(ctrlNames{k}).val))
            defExpand(obj, obj.u.(ctrlNames{k}));
            ctrlDef = remStructs(obj, obj.u.(ctrlNames{k}).def);       
            fprintf(fid, ['\t u(' num2str(k) ') = ' ...
                ctrlDef ';\n']);
        end
    end

    % create struct of auxilary states
    fprintf(fid, '\n\t%% Create struct of auxiliary states\n');
    for k=1:length(auxNames)
        auxDef = remStructs(obj, obj.a.(auxNames{k}).def);       
        fprintf(fid, ['\t a.' auxNames{k} ' = ' ...
            auxDef ';\n']);
    end
    
    % create array of derivatives of states
    fprintf(fid, '\n\t%% Create array of derivatives \n');
    for k=1:length(stateNames)
        odeDef = remStructs(obj, obj.x.(stateNames{k}).def);       
        fprintf(fid, ['\t dx(' num2str(k) ') = ' ...
            odeDef ';\n']);
    end
    
    fprintf(fid, '\n\t dx = dx'';\n');
    
    fprintf(fid, 'end\n');
    
    fclose(fid);
end

function newStr = remStructs(obj, str)
% replace parameter, input, and control names with their array value

    [stateNames, auxNames, ctrlNames, ...
        paramNames, inputNames] = getFieldNames(obj);

    newStr = str;
    % replace all parameter names with their array value
    for n=1:length(paramNames)
        paramLengths(n) = length(paramNames{n});
    end
    [~, paramByLength] = sort(paramLengths,'descend');
    for n=paramByLength
        newStr = strrep(newStr, ['p.' paramNames{n}], ['p(' num2str(n) ')']);
    end

    % replace all input names with their array value
    for n=1:length(inputNames)
        inputLengths(n) = length(inputNames{n});
    end
    [~, inputByLength] = sort(inputLengths,'descend');
    for n=inputByLength
        newStr = strrep(newStr, ['d.' inputNames{n}], ['d(' num2str(n) ')']);
    end

    % replace all predefined control names with their array value
    for n=1:length(ctrlNames)
        ctrlLengths(n) = length(ctrlNames{n});
    end
    [~, ctrlByLength] = sort(ctrlLengths,'descend');
    for n=ctrlByLength
        newStr = strrep(newStr, ['u.' ctrlNames{n}], ['u(' num2str(n) ')']);
    end
    
    % replace all states names with their array value
    for n=1:length(stateNames)
        stateLengths(n) = length(stateNames{n});
    end
    [~, stateByLength] = sort(stateLengths,'descend');
    for n=stateByLength
        newStr = strrep(newStr, ['x.' stateNames{n}], ['x(' num2str(n) ')']);
    end
end